from datetime import date

import streamlit as st

# 定義 Customer 類別
class Customer:
    def __init__(self, name, email, preferences):
        self.name = name
        self.email = email
        self.preferences = preferences

# 定義 RestaurantMenu 類別
class RestaurantMenu:
    def __init__(self, name, main_courses, add_ons):
        self.name = name
        self.main_courses = main_courses
        self.add_ons = add_ons

    def display_menu(self):
        st.write(f"### {self.name} 菜單")
        st.write("#### 主餐:")
        for item, price in self.main_courses.items():
            st.write(f"{item}: {price}元")
        st.write("#### 加點品項:")
        for item, price in self.add_ons.items():
            st.write(f"{item}: {price}元")

# 定義幾個餐廳的菜單
menu1 = RestaurantMenu(
    "餐廳A",
    {
        "雞肉便當": 70,
        "豬肉便當": 75,
        "魚排便當": 85,
    },
    {
        "湯": 10,
        "沙拉": 20,
        "可樂": 15,
    },
)

menu2 = RestaurantMenu(
    "餐廳B",
    {
        "炸雞便當": 80,
        "牛肉便當": 85,
        "豬排便當": 75,
    },
    {
        "白飯（半碗）": 5,
        "白飯（1碗）": 10,
        "滷蛋": 15,
        "豆干": 5,
        "香腸": 45,
        "高麗菜": 35,
    },
)

menu3 = RestaurantMenu(
    "餐廳C",
    {
        "素食便當": 70,
        "烤鴨便當": 95,
        "咖哩雞便當": 85,
    },
    {
        "湯": 10,
        "水果": 20,
        "汽水": 15,
    },
)

# 預設的訂購人員
customers = {
    "Alice": Customer("Alice", "alice@example.com", "無辣"),
    "Bob": Customer("Bob", "bob@example.com", "少鹽"),
    "Charlie": Customer("Charlie", "charlie@example.com", "多辣"),
}

# 餐廳選單
restaurants = {
    "餐廳A": menu1,
    "餐廳B": menu2,
    "餐廳C": menu3,
}

# 使用 Streamlit 構建應用
def main():
    st.title("訂餐系統")
    st.write("選擇餐廳查看菜單並進行訂購:")

    # 選擇訂購人員
    st.sidebar.title("訂購人資訊")
    customer_choice = st.sidebar.selectbox("選擇訂購人員", list(customers.keys()))
    selected_customer = customers[customer_choice]

    # 自動填入訂購人資訊
    customer_name = st.sidebar.text_input(
        "姓名", value=selected_customer.name, disabled=True
    )
    customer_email = st.sidebar.text_input(
        "Email", value=selected_customer.email, disabled=True
    )
    customer_preferences = st.sidebar.text_input(
        "喜好", value=selected_customer.preferences
    )
    order_date = st.sidebar.date_input("餐點日期", date.today())

    # 選擇餐廳
    restaurant_choice = st.selectbox("選擇餐廳", list(restaurants.keys()))
    selected_menu = restaurants[restaurant_choice]

    # 顯示菜單
    selected_menu.display_menu()

    # 選擇主餐和數量
    st.write("\n### 選擇主餐和數量:")
    main_courses_selected = {}
    for item in selected_menu.main_courses.keys():
        quantity = st.number_input(
            f"{item} 數量", min_value=0, max_value=10, step=1, key=f"main_{item}"
        )
        if quantity > 0:
            main_courses_selected[item] = quantity

    # 選擇加點品項和數量
    st.write("\n### 選擇加點品項和數量:")
    add_ons_selected = {}
    for item in selected_menu.add_ons.keys():
        quantity = st.number_input(
            f"{item} 數量", min_value=0, max_value=10, step=1, key=f"add_{item}"
        )
        if quantity > 0:
            add_ons_selected[item] = quantity

    # 計算總價
    total_price = 0
    for item, quantity in main_courses_selected.items():
        total_price += selected_menu.main_courses[item] * quantity
    for item, quantity in add_ons_selected.items():
        total_price += selected_menu.add_ons[item] * quantity

    # 顯示訂單和總價
    if st.button("確認訂單"):
        st.write("### 訂單詳情:")
        st.write(f"姓名: {customer_name}")
        st.write(f"Email: {customer_email}")
        st.write(f"喜好: {customer_preferences}")
        st.write(f"餐點日期: {order_date}")
        st.write("#### 主餐:")
        for item, quantity in main_courses_selected.items():
            st.write(
                f"{item} - {selected_menu.main_courses[item]}元 x {quantity} = {selected_menu.main_courses[item] * quantity}元"
            )
        st.write("#### 加點品項:")
        for item, quantity in add_ons_selected.items():
            st.write(
                f"{item} - {selected_menu.add_ons[item]}元 x {quantity} = {selected_menu.add_ons[item] * quantity}元"
            )
        st.write(f"### 總價: {total_price}元")

if __name__ == "__main__":
    main()
