import dns.resolver

class DNSBLChecker:
    def __init__(self, providers=None):
        """
        初始化 DNSBLChecker，允許自定義 DNSBL 提供者列表。
        :param providers: DNSBL 提供者列表 (可選，默認為常用黑名單提供者)
        """
        self.providers = providers or [
            "zen.spamhaus.org",
            "bl.spamcop.net",
            "dnsbl.sorbs.net",
            "b.barracudacentral.org"
        ]

    def add_provider(self, provider):
        """
        新增一個 DNSBL 提供者。
        :param provider: 提供者的域名，例如 "new.blacklist.com"
        """
        if provider not in self.providers:
            self.providers.append(provider)
            print(f"已新增提供者：{provider}")
        else:
            print(f"提供者 {provider} 已存在。")

    def remove_provider(self, provider):
        """
        移除一個 DNSBL 提供者。
        :param provider: 提供者的域名
        """
        if provider in self.providers:
            self.providers.remove(provider)
            print(f"已移除提供者：{provider}")
        else:
            print(f"提供者 {provider} 不在列表中。")

    def check_ip(self, ip_address):
        """
        檢查 IP 是否被列入任何 DNSBL 提供者的黑名單中。
        :param ip_address: 要檢查的 IP 地址
        """
        reversed_ip = ".".join(reversed(ip_address.split(".")))
        blacklisted_by = []

        for provider in self.providers:
            query = f"{reversed_ip}.{provider}"
            try:
                # 嘗試解析 DNS 查詢
                dns.resolver.resolve(query, "A")
                blacklisted_by.append(provider)
            except dns.resolver.NXDOMAIN:
                # 未列入該提供者黑名單
                continue
            except Exception as e:
                print(f"檢查 {provider} 時發生錯誤：{e}")

        return blacklisted_by

    def display_result(self, ip_address):
        """
        顯示檢查結果。
        :param ip_address: 要檢查的 IP 地址
        """
        blacklisted_by = self.check_ip(ip_address)
        if blacklisted_by:
            print(f"IP 地址 {ip_address} 被列入以下黑名單：")
            for provider in blacklisted_by:
                print(f"- {provider}")
        else:
            print(f"IP 地址 {ip_address} 未被列入任何黑名單。")


if __name__ == "__main__":
    # 初始化 DNSBL 檢查器
    dnsbl_checker = DNSBLChecker()

    # 檢查 IP
    ip_to_check = "210.242.157.193"  # 替換成你的郵件伺服器 IP 地址
    dnsbl_checker.display_result(ip_to_check)

    # 動態操作提供者列表
    dnsbl_checker.add_provider("new.blacklist.com")
    dnsbl_checker.remove_provider("bl.spamcop.net")

