import Adafruit_DHT
import schedule
import time
from gpiozero import LED
import pymysql.cursors
from datetime import datetime
import os
import logging

# 日誌設定
logging.basicConfig(level=logging.INFO, filename='sensor.log',
                    format='%(asctime)s - %(levelname)s - %(message)s')

# 資料庫設定
DB_HOST = os.getenv('DB_HOST', '127.0.0.1')  # 預設為本地端
DB_USER = os.getenv('DB_USER', 'db_user')    # 資料庫使用者名稱
DB_PASSWORD = os.getenv('DB_PASSWORD', 'db_password')  # 資料庫密碼
DB_NAME = os.getenv('DB_NAME', 'environment')  # 資料庫名稱

# LED 腳位設定
green_led = LED(17)  # 綠燈
yellow_led = LED(27)  # 黃燈
red_led = LED(22)  # 紅燈

# DHT11 感測器設定
sensor = Adafruit_DHT.DHT11  # 感測器型號
pin = 4  # 感測器連接的 GPIO 腳位


# LED 控制函數
def control_leds(temp):
    green_led.on() if temp < 25.0 else green_led.off()
    yellow_led.on() if 25.0 <= temp < 30.0 else yellow_led.off()
    red_led.on() if temp >= 30.0 else red_led.off()


# 定時執行的核心任務
def job():
    try:
        # 讀取感測器數據
        humidity, temperature = Adafruit_DHT.read_retry(sensor, pin)
        now = datetime.now().strftime("%Y-%m-%d %H:%M:%S")

        if humidity is not None and temperature is not None:
            logging.info(f"{now} - Temp: {temperature:.1f}°C, Humidity: {humidity:.1f}%")

            # 儲存數據到資料庫
            try:
                connection = pymysql.connect(
                    host=DB_HOST,
                    user=DB_USER,
                    password=DB_PASSWORD,
                    db=DB_NAME,
                    charset='utf8mb4',
                    cursorclass=pymysql.cursors.DictCursor
                )

                with connection.cursor() as cursor:
                    sql = "INSERT INTO environment_data (humidity, temperature) VALUES (%s, %s)"
                    cursor.execute(sql, (humidity, temperature))
                connection.commit()

            except Exception as db_err:
                logging.error(f"資料庫連線錯誤: {db_err}")
            finally:
                if 'connection' in locals():
                    connection.close()

            # 控制 LED
            control_leds(temperature)
        else:
            logging.warning("感測器讀取失敗，無法取得數據")

    except Exception as e:
        logging.error(f"執行任務時發生錯誤: {e}")


# 排程設定，每 5 分鐘執行一次
schedule.every(300).seconds.do(job)

try:
    while True:
        schedule.run_pending()
        time.sleep(1)
except KeyboardInterrupt:
    logging.info("程式已被使用者中斷")
    green_led.off()
    yellow_led.off()
    red_led.off()
