import json
from http.server import BaseHTTPRequestHandler, HTTPServer
from urllib.parse import urlparse, parse_qs
import shutil


class DiskUsage:
    def __init__(self, path="/"):
        self.path = path

    def get_usage(self):
        try:
            total, used, free = shutil.disk_usage(self.path)
            usage_percent = (used / total) * 100 if total > 0 else 0
            return {
                "path": self.path,
                "total_space": total,
                "used_space": used,
                "free_space": free,
                "usage_percent": usage_percent,
            }
        except Exception as e:
            return {"error": str(e)}

    @staticmethod
    def format_size(size_in_bytes):
        for unit in ["B", "KB", "MB", "GB", "TB"]:
            if size_in_bytes < 1024:
                return f"{size_in_bytes:.2f} {unit}"
            size_in_bytes /= 1024
        return f"{size_in_bytes:.2f} PB"


class DiskUsageHandler(BaseHTTPRequestHandler):
    def do_GET(self):
        parsed_path = urlparse(self.path)
        if parsed_path.path == "/disk-usage":
            # 取得 query string 中的 path 參數
            query = parse_qs(parsed_path.query)
            path = query.get("path", ["/"])[0]

            # 執行 DiskUsage 並取得結果
            disk_usage = DiskUsage(path)
            usage = disk_usage.get_usage()

            # 回傳 JSON 結果
            self.send_response(200)
            self.send_header("Content-Type", "application/json")
            self.end_headers()
            self.wfile.write(json.dumps(usage).encode("utf-8"))
        else:
            self.send_response(404)
            self.send_header("Content-Type", "application/json")
            self.end_headers()
            self.wfile.write(json.dumps({"error": "Not Found"}).encode("utf-8"))


def run(server_class=HTTPServer, handler_class=DiskUsageHandler, port=8000):
    server_address = ("", port)
    httpd = server_class(server_address, handler_class)
    print(f"Starting server on port {port}...")
    httpd.serve_forever()


if __name__ == "__main__":
    run()
