#!/bin/bash

# Flush all existing rules
iptables -F

# Set default policies to drop
iptables -P INPUT DROP
iptables -P FORWARD DROP
iptables -P OUTPUT ACCEPT

# Allow established connections
iptables -A INPUT -m state --state ESTABLISHED,RELATED -j ACCEPT -m comment --comment "Allow established connections"

# Allow loopback interface traffic
iptables -A INPUT -i lo -j ACCEPT -m comment --comment "Allow loopback interface"

# Allow SSH traffic (port 22)
# iptables -A INPUT -p tcp --dport 22 -j ACCEPT -m comment --comment "Allow SSH"

# Allow SSH traffic only from IPs in the taiwan_ips ipset collection
iptables -A INPUT -p tcp --dport 22 -m set --match-set taiwan_ips src -j ACCEPT -m comment --comment "Allow SSH from Taiwan IPs"

# Drop SSH traffic from other IPs
iptables -A INPUT -p tcp --dport 22 -j DROP -m comment --comment "Drop SSH from other IPs"


# Allow HTTP traffic (port 80)
# iptables -A INPUT -p tcp --dport 80 -j ACCEPT -m comment --comment "Allow HTTP"

# Allow HTTPS traffic (port 443)
# iptables -A INPUT -p tcp --dport 443 -j ACCEPT -m comment --comment "Allow HTTPS"

# Allow ping traffic
iptables -A INPUT -p icmp --icmp-type echo-request -j ACCEPT -m comment --comment "Allow ping"

# Block SYN Flood attacks
iptables -A INPUT -p tcp --syn -m limit --limit 1/s --limit-burst 4 -j ACCEPT -m comment --comment "Protect against SYN Flood"

# Block Port Scanning
iptables -A INPUT -p tcp --syn -m recent --name scan --set -m comment --comment "Detect port scans"
iptables -A INPUT -p tcp --syn -m recent --name scan --update --seconds 60 --hitcount 10 -j DROP -m comment --comment "Drop port scans"

# Block DoS attacks
iptables -A INPUT -p tcp -m limit --limit 10/s --limit-burst 20 -j ACCEPT -m comment --comment "Limit incoming connections to prevent DoS"
iptables -A INPUT -p udp -m limit --limit 10/s --limit-burst 20 -j ACCEPT -m comment --comment "Limit UDP connections to prevent DoS"
iptables -A INPUT -p icmp -m limit --limit 1/s --limit-burst 3 -j ACCEPT -m comment --comment "Limit ICMP requests to prevent DoS"

# Limit UDP traffic to prevent UDP Flood attack
iptables -A INPUT -p udp -m limit --limit 10/s --limit-burst 20 -j ACCEPT -m comment --comment "Limit UDP traffic to prevent UDP Flood"

# Drop oversized ICMP packets to prevent Ping of Death attack
iptables -A INPUT -p icmp --icmp-type echo-request -m length --length 0:1024 -j ACCEPT -m comment --comment "Drop oversized ICMP packets"

# Drop packets with a source address equal to the broadcast address
iptables -A INPUT -p icmp --icmp-type echo-request -s 255.255.255.255 -j DROP -m comment --comment "Drop ICMP packets with broadcast source address"

# Drop packets with the same source and destination address
iptables -A INPUT -s 192.168.1.1 -d 192.168.1.1 -j DROP -m comment --comment "Drop packets with same source and destination address"

# Drop invalid packets
iptables -A INPUT -m state --state INVALID -j DROP -m comment --comment "Drop invalid packets"

# Drop suspicious port scanning attempts
iptables -A INPUT -p tcp --syn -m recent --name portscan --set -m comment --comment "Track port scanning attempts"
iptables -A INPUT -p tcp --syn -m recent --name portscan --update --seconds 60 --hitcount 10 -j DROP -m comment --comment "Drop excessive port scan attempts"

# Rate limit incoming connections to mitigate DDoS
iptables -A INPUT -p tcp -m limit --limit 20/s --limit-burst 100 -j ACCEPT -m comment --comment "Rate limit TCP connections to mitigate DDoS"
iptables -A INPUT -p udp -m limit --limit 20/s --limit-burst 100 -j ACCEPT -m comment --comment "Rate limit UDP connections to mitigate DDoS"


# Set NAT forwarding rules (if needed)
# iptables -t nat -A POSTROUTING -o eth0 -j MASQUERADE -m comment --comment "NAT outbound traffic"

# Save rules
# iptables-save > /etc/iptables/rules.v4