#!/bin/bash

# === Basic Settings ===

# Clear all existing rules
iptables -F
iptables -X
iptables -Z

# Set default policies: drop all incoming and forwarding traffic, allow outgoing
iptables -P INPUT DROP
iptables -P FORWARD DROP
iptables -P OUTPUT ACCEPT

# === Variables ===
TAIWAN_IPSET="taiwan_ips"  # Name of the IP set for Taiwan
ALLOWED_SSH_IP="192.168.1.0/24"  # Allowed SSH subnet

# === Basic Allow Rules ===

# Allow established and related connections
iptables -A INPUT -m state --state ESTABLISHED,RELATED -j ACCEPT -m comment --comment "Allow established connections"

# Allow loopback (local) traffic
iptables -A INPUT -i lo -j ACCEPT -m comment --comment "Allow local traffic"

# === SSH Rules ===

# Allow SSH from Taiwan IP set
iptables -A INPUT -p tcp --dport 22 -m set --match-set $TAIWAN_IPSET src -j ACCEPT -m comment --comment "Allow SSH from Taiwan IPs"

# Allow SSH from allowed internal network
iptables -A INPUT -p tcp --dport 22 -s $ALLOWED_SSH_IP -j ACCEPT -m comment --comment "Allow SSH from internal network"

# Limit SSH attempts to prevent brute force attacks
iptables -A INPUT -p tcp --dport 22 -m recent --name ssh_bruteforce --set
iptables -A INPUT -p tcp --dport 22 -m recent --name ssh_bruteforce --update --seconds 60 --hitcount 5 -j DROP -m comment --comment "Limit SSH attempts"

# Drop SSH from other sources
iptables -A INPUT -p tcp --dport 22 -j DROP -m comment --comment "Drop unauthorized SSH traffic"

# === HTTP/HTTPS Traffic ===

# Allow HTTP (port 80)
iptables -A INPUT -p tcp --dport 80 -j ACCEPT -m comment --comment "Allow HTTP traffic"

# Allow HTTPS (port 443)
iptables -A INPUT -p tcp --dport 443 -j ACCEPT -m comment --comment "Allow HTTPS traffic"

# === Attack Prevention ===

# Protect against SYN Flood
iptables -A INPUT -p tcp --syn -m limit --limit 1/s --limit-burst 4 -j ACCEPT -m comment --comment "Protect from SYN Flood"

# Detect and drop port scans
iptables -A INPUT -p tcp --syn -m recent --name portscan --set -m comment --comment "Detect port scans"
iptables -A INPUT -p tcp --syn -m recent --name portscan --update --seconds 60 --hitcount 10 -j DROP -m comment --comment "Drop port scans"

# Limit TCP, UDP, and ICMP traffic to prevent DoS attacks
iptables -A INPUT -p tcp -m limit --limit 10/s --limit-burst 20 -j ACCEPT -m comment --comment "Limit TCP traffic for DoS"
iptables -A INPUT -p udp -m limit --limit 10/s --limit-burst 20 -j ACCEPT -m comment --comment "Limit UDP traffic for DoS"
iptables -A INPUT -p icmp -m limit --limit 1/s --limit-burst 3 -j ACCEPT -m comment --comment "Limit ICMP requests for DoS"

# Drop large ICMP packets to prevent Ping of Death
iptables -A INPUT -p icmp --icmp-type echo-request -m length --length 0:1024 -j ACCEPT -m comment --comment "Prevent Ping of Death"

# Drop packets with the same source and destination IP
iptables -A INPUT -s 192.168.1.1 -d 192.168.1.1 -j DROP -m comment --comment "Drop same source and destination"

# Drop invalid packets
iptables -A INPUT -m state --state INVALID -j DROP -m comment --comment "Drop invalid packets"

# === Logging ===

# Log dropped packets
iptables -A INPUT -m limit --limit 2/min -j LOG --log-prefix "Dropped: " --log-level 4 -m comment --comment "Log dropped packets"

# === Save Rules ===

# Save rules to /etc/iptables/rules.v4
iptables-save > /etc/iptables/rules.v4
