import requests

class HolidayAPI:
    def __init__(self, base_url):
        """
        Initialize the HolidayAPI class.
        :param base_url: The base URL for the API endpoint.
        """
        self.base_url = base_url

    def get_holidays(self, page=0, size=10):
        """
        Fetch holiday data from the API.
        :param page: The page number to fetch.
        :param size: The number of records per page.
        :return: A list of holiday data dictionaries.
        """
        url = f"{self.base_url}?page={page}&size={size}"
        headers = {
            'accept': 'application/json'
        }
        response = requests.get(url, headers=headers)

        if response.status_code == 200:
            return response.json()
        else:
            response.raise_for_status()

    def get_holidays_by_year(self, year):
        """
        Fetch holiday data for a specific year.
        :param year: The year for which to fetch holidays.
        :return: A list of holiday data dictionaries for the given year.
        """
        all_holidays = []
        page = 0
        while True:
            data = self.get_holidays(page=page, size=100)
            if not data:
                break
            all_holidays.extend(data)
            page += 1

        return [holiday for holiday in all_holidays if holiday.get("year") == str(year)]

    def is_holiday(self, date):
        """
        Check if a specific date is a holiday.
        :param date: The date to check in YYYYMMDD format.
        :return: True if the date is a holiday, False otherwise.
        """
        page = 0
        while True:
            data = self.get_holidays(page=page, size=100)
            if not data:
                break
            for holiday in data:
                if holiday.get("date") == date and holiday.get("isholiday") == "是":
                    return True
            page += 1
        return False

class Holiday:
    def __init__(self, date, year, name, isholiday, holidaycategory, description):
        """
        Represent a single holiday entry.
        :param date: The date of the holiday in YYYYMMDD format.
        :param year: The year of the holiday.
        :param name: The name of the holiday (if any).
        :param isholiday: Whether it is a holiday ('是' or '否').
        :param holidaycategory: The category of the holiday.
        :param description: The description of the holiday.
        """
        self.date = date
        self.year = year
        self.name = name
        self.isholiday = isholiday
        self.holidaycategory = holidaycategory
        self.description = description

    def __repr__(self):
        return f"Holiday(date={self.date}, year={self.year}, name={self.name}, " \
               f"isholiday={self.isholiday}, holidaycategory={self.holidaycategory}, " \
               f"description={self.description})"

# Example usage
if __name__ == "__main__":
    api = HolidayAPI("https://data.ntpc.gov.tw/api/datasets/308dcd75-6434-45bc-a95f-584da4fed251/json")
    holidays_2025 = api.get_holidays_by_year(2025)

    holidays = []
    for entry in holidays_2025:
        holiday = Holiday(
            date=entry.get("date"),
            year=entry.get("year"),
            name=entry.get("name"),
            isholiday=entry.get("isholiday"),
            holidaycategory=entry.get("holidaycategory"),
            description=entry.get("description")
        )
        holidays.append(holiday)

    for holiday in holidays:
        print(holiday)

    # Check if a specific date is a holiday
    # specific_date = "20250101"
    specific_date = "20250102"
    print(f"Is {specific_date} a holiday? {api.is_holiday(specific_date)}")

