#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
constellation.py: Description of what this script does.

Author: Timmy
Copyright: Copyright 2022, Timmy
License: MIT
Version: 1.0
"""


from datetime import datetime

# 嘗試載入 icecream 套件中的 ic 函數
try:
    from icecream import ic

    # 配置 ic 函數的輸出，包括上下文訊息
    ic.configureOutput(includeContext=True)

# 如果 icecream 套件沒有安裝，則會產生 ImportError
except ImportError:
    # 如果沒有安裝 icecream 套件，則定義一個名為 ic 的 lambda 函數
    # 這個函數的作用是在沒有載入 icecream 套件時，提供一個替代的方式來輸出訊息和變數
    ic = lambda *a: None if not a else (a[0] if len(a) == 1 else a)  # noqa


class Constellation:
    def __init__(self):
        """
        Aries: March 21-April 19
        Taurus: April 20-May 20
        Gemini: May 21-June 20
        Cancer: June 21-July 22
        Leo: July 23-August 22
        Virgo: August 23-September 22
        Libra: September 23-October 22
        Scorpio: October 23-November 21
        Sagittarius: November 22-December 21
        Capricorn: December 22-January 19
        Aquarius: January 20-February 18
        Pisces: February 19-March 20
        """

        # 定義每個星座的日期範圍
        self.constellations = [
            ("摩羯座", "12-22", "01-19"),
            ("水瓶座", "01-20", "02-18"),
            ("雙魚座", "02-19", "03-20"),
            ("牡羊座", "03-21", "04-19"),
            ("金牛座", "04-20", "05-20"),
            ("雙子座", "05-21", "06-20"),
            ("巨蟹座", "06-21", "07-22"),
            ("獅子座", "07-23", "08-22"),
            ("處女座", "08-23", "09-22"),
            ("天秤座", "09-23", "10-22"),
            ("天蠍座", "10-23", "11-21"),
            ("射手座", "11-22", "12-21"),
        ]

    def get_month_and_day(self, date):
        """
        這個函式會接收一個日期，並回傳這個日期的月份和日期。

        Args:
            date (str): 一個日期，格式為 YYYY-MM-DD。

        Returns:
            tuple: 一個元組，包含月份和日期。
        """

        # 把日期轉換為日期物件
        date_obj = datetime.strptime(date, "%Y-%m-%d")

        # 取得月份和日期
        month = date_obj.month
        day = date_obj.day

        # 回傳月份和日期
        return (month, day)

    def get_constellation(self, date):
        """
        這個函數的作用是根據輸入的日期，返回該日期所屬的星座。

        Args:
            date (str): 一個日期，格式為 YYYY-MM-DD。

        Returns:
            str: 返回星座的名稱
        """

        # 取得月份和日期
        month, day = self.get_month_and_day(date)

        # 把傳入的月份和日期轉換為日期物件
        birth_date = datetime.strptime(f"{month}-{day}", "%m-%d")

        # 找出生日所屬的星座
        for constellation in self.constellations:
            name, start, end = constellation
            start_date = datetime.strptime(start, "%m-%d")
            end_date = datetime.strptime(end, "%m-%d")
            # if birth_date >= start_date and birth_date <= end_date:
            #     return name
            if start_date <= birth_date <= end_date:
                return name

        # 如果沒找到，則回傳空字串
        return ""


# 檢查當前程式是否被作為主程式執行
if __name__ == "__main__":

    # 建立 Constellation 類別的物件
    constellation = Constellation()

    # 使用 get_constellation 方法查詢 5 月 25 日出生的人所屬的星座
    result = constellation.get_constellation("1995-05-25")
    print(result)  # 輸出 '雙子座'

    result = constellation.get_constellation("1977-10-12")
    print(result)  # 輸出 '天秤座'

    result = constellation.get_constellation("1981-08-29")
    print(result)  # 輸出 '處女座'

    result = constellation.get_constellation("2016-02-16")
    print(result)  # 輸出 '水瓶座'
