import datetime

class LoggingMixin:
    """ 提供日誌功能的 Mixin 類別 """
    
    def log(self, message):
        """ 簡單的日誌紀錄方法 """
        timestamp = datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        print(f"[{timestamp}] {self.__class__.__name__}: {message}")

class TimestampMixin:
    """ 提供時間標記功能的 Mixin 類別 """
    
    def __init__(self, *args, **kwargs):
        """ 設定物件的建立與更新時間 """
        self.created_at = datetime.datetime.now()
        self.updated_at = self.created_at
        super().__init__(*args, **kwargs)

    def update_timestamp(self):
        """ 更新 updated_at 時間 """
        self.updated_at = datetime.datetime.now()
        self.log(f"更新時間標記: {self.updated_at}")

class User(LoggingMixin, TimestampMixin):
    """ 使用 LoggingMixin 和 TimestampMixin 的 User 類別 """
    
    def __init__(self, name):
        super().__init__()  # 確保 Mixin 初始化
        self.name = name
        self.log(f"User '{self.name}' 已建立，建立時間: {self.created_at}")

    def update_name(self, new_name):
        self.log(f"名稱從 '{self.name}' 變更為 '{new_name}'")
        self.name = new_name
        self.update_timestamp()

# 測試範例
if __name__ == "__main__":
    user = User("Alice")  # 會自動記錄 "User 'Alice' 已建立"
    user.update_name("Bob")  # 會記錄名稱變更並更新時間標記
