#!/usr/bin/env bash
# 用途：查看 Kubernetes 服務狀態與實際連線方式
# 包含：Pod、Service(NodePort)、Ingress(HTTP/HTTPS)、PVC
# 適用：k3s + Traefik

set -euo pipefail

echo
echo "=== Pods ==="
kubectl get pods -o wide

echo
echo "=== Services ==="
kubectl get svc

echo
echo "=== Node IPs ==="
kubectl get nodes -o jsonpath='{range .items[*]}{.metadata.name}{"\t"}{range .status.addresses[?(@.type=="InternalIP")]}{.address}{" "}{end}{"\n"}{end}'

echo
echo "=== NodePort Access ==="
NODE_IPS=$(kubectl get nodes -o jsonpath='{range .items[*]}{.status.addresses[?(@.type=="InternalIP")].address}{"\n"}{end}')
NODEPORT_SVCS=$(kubectl get svc -o jsonpath='{range .items[?(@.spec.type=="NodePort")]}{.metadata.name}{"\t"}{.spec.ports[0].nodePort}{"\n"}{end}')

if [ -z "$NODEPORT_SVCS" ]; then
  echo "（無 NodePort 服務）"
else
  for ip in $NODE_IPS; do
    while IFS=$'\t' read -r svc port; do
      echo "- $ip:$port ($svc)"
    done <<< "$NODEPORT_SVCS"
  done
fi

echo
echo "=== Ingress (Access URLs) ==="
INGRESS_LIST=$(kubectl get ingress -o name 2>/dev/null || true)

if [ -z "$INGRESS_LIST" ]; then
  echo "（無 Ingress）"
else
  for ing in $INGRESS_LIST; do
    NAME=$(kubectl get "$ing" -o jsonpath='{.metadata.name}')
    HOSTS=$(kubectl get "$ing" -o jsonpath='{range .spec.rules[*]}{.host}{"\n"}{end}')
    TLS=$(kubectl get "$ing" -o jsonpath='{.spec.tls}')

    echo "- $NAME"
    while read -r host; do
      [ -z "$host" ] && continue
      if [ -n "$TLS" ]; then
        echo "  https://$host"
      else
        echo "  http://$host"
      fi
    done <<< "$HOSTS"
  done
fi

echo
echo "=== PVC ==="
kubectl get pvc

