class SeverancePayCalculator:
    """
    Legendary 級資遣費計算器，適用於新制與舊制。
    - 自動計算應得資遣費
    - 內建異常處理與日誌紀錄
    """
    
    def __init__(self, salary, years_of_service, policy="new", log=True):
        """
        初始化資遣費計算器
        :param salary: 每月工資 (int or float)
        :param years_of_service: 服務年數 (float)
        :param policy: "new" (新制) 或 "old" (舊制)
        :param log: 是否啟用日誌 (bool)
        """
        self.salary = salary
        self.years_of_service = years_of_service
        self.policy = policy.lower()
        self.log_enabled = log
        self.log(f"初始化: 工資={salary}, 年資={years_of_service}, 政策={policy}")
    
    def log(self, message):
        """ 簡單的日誌紀錄 """
        if self.log_enabled:
            from datetime import datetime
            timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
            print(f"[{timestamp}] {self.__class__.__name__}: {message}")
    
    def calculate(self):
        """ 計算資遣費 """
        try:
            if self.policy == "new":
                severance_pay = min(self.years_of_service * 0.5, 6) * self.salary
            elif self.policy == "old":
                severance_pay = min(self.years_of_service, 6) * self.salary
            else:
                raise ValueError("政策類型錯誤，請選擇 'new' 或 'old'")
            
            result = round(severance_pay, 2)
            self.log(f"計算結果: {result} 元")
            return result
        except Exception as e:
            self.log(f"錯誤: {e}")
            return None
    
    def __str__(self):
        return f"{self.policy.upper()} 制資遣費: {self.calculate()} 元"

# 測試範例
if __name__ == "__main__":
    salary = 100000  # 每月工資
    years_of_service = 8.5  # 服務年數

    new_policy_calculator = SeverancePayCalculator(salary, years_of_service, "new")
    old_policy_calculator = SeverancePayCalculator(salary, years_of_service, "old")

    print(new_policy_calculator)
    print(old_policy_calculator)
