import datetime
import json
import traceback

class LoggingMixin:
    """ 提供日誌功能的 Mixin 類別 """
    
    def log(self, message):
        """ 記錄日誌 """
        timestamp = datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        print(f"[{timestamp}] {self.__class__.__name__}: {message}")

class TimestampMixin:
    """ 提供時間標記功能的 Mixin 類別 """
    
    def __init__(self, *args, **kwargs):
        """ 設定物件的建立與更新時間 """
        self.created_at = datetime.datetime.now()
        self.updated_at = self.created_at
        super().__init__(*args, **kwargs)

    def update_timestamp(self):
        """ 更新 updated_at 時間 """
        self.updated_at = datetime.datetime.now()
        self.log(f"更新時間標記: {self.updated_at}")

class DatabaseMixin:
    """ 模擬資料儲存功能的 Mixin 類別，支援 JSON 序列化 """
    _database = {}

    def save(self):
        """ 儲存物件至模擬資料庫 """
        DatabaseMixin._database[self.name] = json.dumps(self.__dict__, default=str)
        self.log(f"資料已儲存: {self.name}")

    @classmethod
    def get(cls, name):
        """ 從資料庫中獲取物件 """
        data = cls._database.get(name, None)
        if data:
            return json.loads(data)
        return None

class PermissionMixin:
    """ 提供權限管理的 Mixin 類別 """
    
    def __init__(self, permissions=None, *args, **kwargs):
        """ 初始化權限 """
        self.permissions = permissions if permissions else []
        super().__init__(*args, **kwargs)

    def has_permission(self, perm):
        """ 檢查是否擁有指定權限 """
        return perm in self.permissions

    def grant_permission(self, perm):
        """ 賦予權限 """
        if perm not in self.permissions:
            self.permissions.append(perm)
            self.log(f"已新增權限: {perm}")

    def revoke_permission(self, perm):
        """ 移除權限 """
        if perm in self.permissions:
            self.permissions.remove(perm)
            self.log(f"已移除權限: {perm}")

class EventMixin:
    """ 提供事件監聽功能的 Mixin 類別 """
    _events = {}

    @classmethod
    def on(cls, event_name, callback):
        """ 訂閱事件 """
        if event_name not in cls._events:
            cls._events[event_name] = []
        cls._events[event_name].append(callback)

    @classmethod
    def trigger(cls, event_name, *args, **kwargs):
        """ 觸發事件 """
        if event_name in cls._events:
            for callback in cls._events[event_name]:
                callback(*args, **kwargs)

class ExceptionHandlingMixin:
    """ 提供異常處理的 Mixin 類別 """
    
    def handle_exception(self, func, *args, **kwargs):
        """ 包裝方法，捕獲異常並記錄 """
        try:
            return func(*args, **kwargs)
        except Exception as e:
            self.log(f"發生異常: {e}")
            traceback.print_exc()

class User(LoggingMixin, TimestampMixin, DatabaseMixin, PermissionMixin, EventMixin, ExceptionHandlingMixin):
    """ User 類別，擁有日誌、時間標記、資料儲存、權限管理、事件監聽、異常處理功能 """

    def __init__(self, name, permissions=None):
        super().__init__(permissions=permissions)
        self.name = name
        self.log(f"User '{self.name}' 已建立，建立時間: {self.created_at}")

    def update_name(self, new_name):
        """ 更新使用者名稱，並觸發事件 """
        self.log(f"名稱從 '{self.name}' 變更為 '{new_name}'")
        old_name = self.name
        self.name = new_name
        self.update_timestamp()
        self.trigger("user_updated", old_name, new_name)

# 測試範例
if __name__ == "__main__":
    # 訂閱使用者更新事件
    def on_user_updated(old_name, new_name):
        print(f"🔔 使用者名稱變更通知: {old_name} → {new_name}")

    User.on("user_updated", on_user_updated)

    # 建立使用者 Alice，並設定權限
    user1 = User("Alice", permissions=["read", "write"])
    user1.save()  # 儲存到模擬資料庫
    
    # 更新名稱
    user1.update_name("Alice_Wonderland")

    # 檢查權限
    print(f"Alice 是否有 'read' 權限？{user1.has_permission('read')}")
    print(f"Alice 是否有 'delete' 權限？{user1.has_permission('delete')}")

    # 賦予新權限
    user1.grant_permission("delete")
    
    # 從資料庫取得使用者
    retrieved_user = User.get("Alice_Wonderland")
    if retrieved_user:
        print(f"從資料庫取得使用者: {retrieved_user}")

    # 測試異常處理
    user1.handle_exception(lambda: 1 / 0)  # 觸發除零錯誤
