import time
import yaml
import json
from abc import ABC, abstractmethod
import streamlit as st

# 策略介面
class DataHandlerStrategy(ABC):
    @abstractmethod
    def read_data(self, path):
        pass

    @abstractmethod
    def write_data(self, path, data):
        pass

# YAML 策略
class YamlHandler(DataHandlerStrategy):
    def read_data(self, path):
        with open(path, "r") as f:
            return yaml.safe_load(f)

    def write_data(self, path, data):
        with open(path, "w") as f:
            yaml.safe_dump(data, f)

# JSON 策略
class JsonHandler(DataHandlerStrategy):
    def read_data(self, path):
        with open(path, "r") as f:
            return json.load(f)

    def write_data(self, path, data):
        with open(path, "w") as f:
            json.dump(data, f, ensure_ascii=False, indent=4)

# DataManager 類別
class DataManager:
    def __init__(self, data_handler: DataHandlerStrategy):
        self.data_handler = data_handler

    def load_data(self, path):
        """
        載入資料
        """
        return self._load_file(path)

    def save_data(self, path, data):
        """
        儲存資料
        """
        try:
            self.data_handler.write_data(path, data)
            st.success("資料已成功儲存")
            time.sleep(1)
            st.rerun()
        except Exception as e:
            st.error(f"儲存資料過程中發生錯誤：{e}")

    def _load_file(self, path):
        """
        通用的文件載入方法
        """
        try:
            return self.data_handler.read_data(path)
        except FileNotFoundError:
            st.error(f"找不到文件：{path}")
        except Exception as e:
            st.error(f"載入文件 {path} 時發生錯誤：{e}")
        return None

# 使用 YAML 策略
yaml_handler = YamlHandler()
data_manager_yaml = DataManager(yaml_handler)
data_yaml = data_manager_yaml.load_data("path/to/your/yamlfile.yaml")
data_manager_yaml.save_data("path/to/your/yamlfile.yaml", data_yaml)

# 使用 JSON 策略
json_handler = JsonHandler()
data_manager_json = DataManager(json_handler)
data_json = data_manager_json.load_data("path/to/your/jsonfile.json")
data_manager_json.save_data("path/to/your/jsonfile.json", data_json)
